

# Addressableアセット 

## 目的と範囲 

本ドキュメントでは、コードベースにおけるUnityのAddressablesシステムの実装、特に`AddressableLoader`ラッパークラスとそのビルトインゲームアセットの読み込み使用について説明します。Addressablesは、直接参照ではなくアドレス文字列でアセットを読み込むコンテンツ管理システムを提供し、グループベースのライフサイクル制御による効率的なメモリ管理を実現します。

ユーザー提供の外部コンテンツ（楽曲、譜面、カスタムスキン）の読み込みについては、[外部コンテンツ読込](#9.2)を参照してください。読み込まれたアセットが冗長な読み込みを避けるためにどのようにキャッシュされるかの詳細については、[キャッシュシステム](#9.3)を参照してください。

---

## システム概要 

Addressableアセットシステムは、UIテクスチャ、プレハブ、デフォルトアセットなどのビルトインゲームリソースの集中的な読み込みとライフサイクル管理を提供します。この実装は、UnityのネイティブAddressables APIをグループベースの参照カウントシステムでラップし、シーンがアセットをまとまった単位として読み込んで解放できるようにします。

### アーキテクチャ図 

```mermaid
flowchart TD

ADDR["_AddessableResources/ ビルトインアセット"]
LOAD["LoadAsync<T>()"]
RELEASE["ReleaseGroup()"]
HANDLE_LIST["_assetGroupHandleList"]
AGH["AssetGroupHandle GroupId + Handle"]
FOLDER["FolderSelectPanel"]
FOLDER_TEX["FolderTextureLoader"]
OTHER["その他のコンポーネント"]

ADDR -.->|"groupId = GetHashCode()"| LOAD
LOAD -.-> AGH
AGH -.-> HANDLE_LIST
FOLDER -.->|"groupId = GetHashCode()"| LOAD
OTHER -.->|"groupId = hash"| LOAD
FOLDER -.->|"OnDestroy()"| RELEASE

subgraph アセット利用者 ["アセット利用者"]
    FOLDER
    FOLDER_TEX
    OTHER
end

subgraph グループハンドル ["グループハンドル"]
    AGH
end

subgraph AddressableLoader ["AddressableLoader"]
    LOAD
    RELEASE
    HANDLE_LIST
end

subgraph アセットソース ["アセットソース"]
    ADDR
end
```

## コアコンポーネント 

### AddressableLoaderクラス 

`AddressableLoader`静的クラスは、Addressableアセットの読み込みと解放の主要なAPIを提供します。呼び出し元が定義したグループIDでグループ化されたすべての読み込まれたアセットハンドルのリストを保持します。

**クラス構造:**

| メンバー | 型 | 目的 |
| --- | --- | --- |
| `_assetGroupHandleList` | `List<AssetGroupHandle>` | すべての読み込まれたアセットをグループの関連付けと共に追跡 |
| `LoadAsync<T>()` | 静的メソッド | アセットを非同期で読み込み、グループに登録 |
| `ReleaseGroup()` | 静的メソッド | 特定のグループIDに関連付けられたすべてのアセットを解放 |

### AssetGroupHandle内部クラス 

`AssetGroupHandle`は、読み込まれた各アセットをグループ識別子に関連付け、バッチ解放操作を可能にします。

```mermaid
classDiagram
    class AddressableLoader {
        -List<AssetGroupHandle> _assetGroupHandleList
        +LoadAsync<T>(CancellationToken ct, string address, int groupId) : T
        +ReleaseGroup(int groupId) : void
    }
    class AssetGroupHandle {
        +int GroupId
        +AsyncOperationHandle Handle
        +AssetGroupHandle(int groupId, AsyncOperationHandle handle)
    }
    AddressableLoader --> AssetGroupHandle : 管理
```

## 読み込みメカニズム 

### LoadAsyncメソッド 

`LoadAsync<T>`メソッドは、自動グループ登録とキャンセルサポート付きの非同期アセット読み込みを提供します。

**メソッドシグネチャ:**

```
public static async UniTask<T> LoadAsync<T>(    
    CancellationToken ct,     
    string assetAddress,     
    int groupId
) where T : UnityEngine.Object
```

**パラメータ:**

| パラメータ | 型 | 説明 |
| --- | --- | --- |
| `ct` | `CancellationToken` | 操作のキャンセルを有効化 |
| `assetAddress` | `string` | Addressableアセットアドレス（例：`AddressableAssetAddress`からの定数） |
| `groupId` | `int` | ライフサイクル管理のためのグループ識別子、通常は呼び出し元の`GetHashCode()` |

**処理フロー:**

```mermaid
sequenceDiagram
  participant p1 as 呼び出し元
  participant p2 as AddressableLoader
  participant p3 as UnityAddressables
  participant p4 as HandleList

  p1->>p2: LoadAsync<T>(ct | address | groupId)
  p2->>p3: LoadAssetAsync<T>(address)
  p3-->>p2: AsyncOperationHandle
  p2->>p4: Add AssetGroupHandle(groupId | handle)
  p2->>p2: await handle.Task
  p2-->>p1: アセットTを返す
  note over p2: キャンセルまたはエラー時、 default(T)を返す
```

**実装詳細:**



このメソッドは:

1. Unityの`Addressables.LoadAssetAsync<T>()`経由で`AsyncOperationHandle`を作成
2. 提供された`groupId`でハンドルを`AssetGroupHandle`にラップ
3. 追跡のためにハンドルを`_assetGroupHandleList`に追加
4. `AttachExternalCancellation(ct)`経由でキャンセルサポート付きで非同期操作を待機
5. 読み込まれたアセットまたはエラー/キャンセル時に`default(T)`を返す

---

## グループベースのライフサイクル管理 

### ReleaseGroupメソッド 

アセットはグループIDごとにバッチで解放され、シーンやコンポーネントが単一の呼び出しですべてのアセットをクリーンアップできます。

**メソッドシグネチャ:**

```c#
public static void ReleaseGroup(int groupId)
```

**解放プロセス:**

```mermaid
sequenceDiagram
  participant p1 as コンポーネント
  participant p2 as AddressableLoader
  participant p3 as UnityAddressables
  participant p4 as HandleList

  p1->>p2: ReleaseGroup(groupId)
  p2->>p4: FindAll(x => x.GroupId == groupId)
  loop 一致する各ハンドル
    p2->>p3: Release(handle.Handle)
  end
  p2->>p4: RemoveAll(x => x.GroupId == groupId)
  p2-->>p1: 解放完了
```

**実装:**



このメソッドは:

1. 提供された`groupId`に一致するすべての`AssetGroupHandle`インスタンスを検索
2. 各ハンドルに対して`Addressables.Release()`を呼び出し
3. `_assetGroupHandleList`から一致するすべてのハンドルを削除

---

## 使用パターン 

### パターン1: コンポーネントスコープの読み込み 

コンポーネントは`GetHashCode()`をグループIDとして使用し、コンポーネントが破棄されたときの自動クリーンアップを保証します。

**FolderTextureLoaderからの例:**

```mermaid
sequenceDiagram
  participant p1 as FolderTextureLoader
  participant p2 as AddressableLoader

  note over p1: LoadTextureAsync()
  p1->>p2: LoadAsync<Texture>(ct | TEXTURENO_IMAGE | GetHashCode())
  p2-->>p1: _noImageTexture
  note over p1: OnFinalize()
  p1->>p2: ReleaseGroup(GetHashCode())
```

### パターン2: シーンスコープの読み込み 

シーンはハッシュ値（多くの場合`_hash => this.GetHashCode()`）をグループIDとして使用し、`OnDestroy()`ですべてのアセットを解放します。

## アセット構成 

### Addressableリソースディレクトリ 

ビルトインのAddressableアセットは、プロジェクトルートの`_AddessableResources/`ディレクトリに保存されます。このディレクトリ構造は、UnityのAddressablesビルドシステムによってスキャンされます。

**ディレクトリ構造:**

```
_AddessableResources/
└── Texture/
    └── SelectMusic/
        ├── all_songs.png
        ├── no_group.png
        └── no_image.png
```

### AddressableAssetAddress定数 

アセットアドレスは定数として定義されます（表示されたファイルには提供されていませんが参照されています）。コードベースは`LoadAsync()`を呼び出す際にこれらの定数を使用します:

| 定数 | 使用目的 | アセットタイプ |
| --- | --- | --- |
| `AddressableAssetAddress.TEXTUREALL_SONGS` | 「All Songs」のデフォルトフォルダサムネイル | Texture |
| `AddressableAssetAddress.TEXTURENO_GROUP` | 「No Group」のデフォルトフォルダサムネイル | Texture |
| `AddressableAssetAddress.TEXTURENO_IMAGE` | フォルダにサムネイルがない場合のフォールバック画像 | Texture |

**使用例:**



```c#
var allSongsTexture = await AddressableLoader.LoadAsync<Texture>(
    ct, 
    AddressableAssetAddress.TEXTUREALL_SONGS, 
    _hash
);

_folderItemList.Add(
    new FolderItem(
        Constant.GroupFolder.ALL_SONGS, 
        folderSongsCountList[0], 
        allSongsTexture
    )
);

if (hasNoGroup)
{    
    var noGroupTexture = await AddressableLoader.LoadAsync<Texture>(
        ct, 
        AddressableAssetAddress.TEXTURENO_GROUP, 
        _hash
    );

    _folderItemList.Add(
        new FolderItem(
            Constant.GroupFolder.NO_GROUP, 
            folderSongsCountList[1], 
            noGroupTexture)
    );
}
```

## 他システムとの統合 

### 外部コンテンツ読込との関係 

Addressableアセットシステムは**ビルトイン**アセットを扱い、外部コンテンツ読込（[外部コンテンツ読込](#9.2)で説明）は**ユーザー提供**アセットを扱います:

| システム | アセットソース | 例 | ローダークラス |
| --- | --- | --- | --- |
| Addressables | `_AddessableResources/` | UIテクスチャ、デフォルト画像、プレハブ | `AddressableLoader` |
| 外部 | ファイルシステム（Songsフォルダなど） | 楽曲ジャケット、譜面、カスタムスキン | `TextureLoader`, `SongInfoLoader` |

**ハイブリッド使用例:**



`FolderTextureLoader`は両方のシステムを組み合わせて使用します:

1. `TextureLoader.Load(path)`を使用して外部ファイルシステムからフォルダ固有の画像を読み込み
2. フォルダにカスタム画像がない場合、Addressableの「no image」テクスチャにフォールバック

```mermaid
flowchart TD

FolderTextureLoader["FolderTextureLoader"]
FileSystem["外部ファイルシステム (Songs/FolderName/)"]
Addressables["Addressables (NO_IMAGE texture)"]

FolderTextureLoader -.->|"カスタム画像あり"| FileSystem
FolderTextureLoader -.->|"カスタム画像なし"| Addressables
```
### キャッシュとの関係 

`AddressableLoader`は独自のキャッシュを実装していませんが（UnityのAddressablesシステムが内部的に処理）、読み込まれたアセットは[キャッシュシステム](#9.3)で文書化されたキャッシュクラスに保存されることがよくあります:

* `FolderTextureCache` - Addressablesとファイルシステムの両方経由で読み込まれたフォルダテクスチャをキャッシュ
* `NoteSkinCache` - カスタムノートスキンスプライトをキャッシュ
* `TouchSeCache` - カスタムタッチサウンドエフェクトをキャッシュ

## エラーハンドリング 

`LoadAsync`メソッドには包括的なエラーハンドリングが含まれています:



```c#
catch (OperationCanceledException ex)
{    
    Debug.LogWarning(ex.Message);
}
catch (Exception ex)
{    
    Debug.LogError(ex.Message);
}
return default(T);
```

**動作:**

* **キャンセル:** 警告をログに記録し、`default(T)`（参照型の場合はnull）を返す
* **その他のエラー:** エラーをログに記録し、`default(T)`を返す
* **成功:** 型`T`の読み込まれたアセットを返す

呼び出し元はnull結果をチェックする必要があります:

```c#
if (!path.HasValue() && _noImageTexture is null)
```

## サマリーテーブル 

| 側面 | 実装 |
| --- | --- |
| **メインクラス** | `AddressableLoader`（静的） |
| **読み込みメソッド** | `LoadAsync<T>(CancellationToken, string, int)` |
| **解放メソッド** | `ReleaseGroup(int)` |
| **グループ化戦略** | 呼び出し元の`GetHashCode()`をグループIDとして使用 |
| **クリーンアップパターン** | `OnDestroy()`または`OnFinalize()`が`ReleaseGroup()`を呼び出し |
| **アセット配置** | `_AddessableResources/`ディレクトリ |
| **アドレス形式** | 文字列定数（例：`AddressableAssetAddress.TEXTUREALL_SONGS`） |
| **並行性** | UniTaskベースのasync/await、キャンセルサポート付き |
| **エラーハンドリング** | エラーまたはキャンセル時に`default(T)`を返す |

### On this page

* [Addressableアセット](#9.1-addressable)
* [目的と範囲](#9.1-)
* [システム概要](#9.1--1)
* [アーキテクチャ図](#9.1--2)
* [コアコンポーネント](#9.1--3)
* [AddressableLoaderクラス](#9.1-addressableloader)
* [AssetGroupHandle内部クラス](#9.1-assetgrouphandle)
* [読み込みメカニズム](#9.1--4)
* [LoadAsyncメソッド](#9.1-loadasync)
* [グループベースのライフサイクル管理](#9.1--5)
* [ReleaseGroupメソッド](#9.1-releasegroup)
* [使用パターン](#9.1--6)
* [パターン1: コンポーネントスコープの読み込み](#9.1-1-)
* [パターン2: シーンスコープの読み込み](#9.1-2-)
* [アセット構成](#9.1--7)
* [Addressableリソースディレクトリ](#9.1-addressable-1)
* [AddressableAssetAddress定数](#9.1-addressableassetaddress)
* [他システムとの統合](#9.1--8)
* [外部コンテンツ読込との関係](#9.1--9)
* [キャッシュとの関係](#9.1--10)
* [エラーハンドリング](#9.1--11)
* [サマリーテーブル](#9.1--12)

