# MVPパターンの実装

## 目的と範囲

このドキュメントは、DKLIKEリズムゲームのコードベース全体で使用されているModel-View-Presenter（MVP）アーキテクチャパターンの実装について説明します。基本クラス階層、ライフサイクル管理、コンストラクタインジェクションのアプローチ、および具体的なシーン実装をカバーしています。シーン固有のビジネスロジックとデータフローの詳細については、[シーンフローとライフサイクル](#2.1)を参照してください。永続化とデータ管理パターンについては、[データ永続化システム](#2.3)を参照してください。

---

## パターン概要

このコードベースは、UI表示（`View`）、ビジネスロジック（`Presenter`）、データ管理（`Model`）の間で関心事を分離するため、MVPパターンの変種を実装しています。このアーキテクチャは以下を提供します：

* **テスタビリティ**: PresenterはUnityの依存関係なしでユニットテスト可能
* **明確な関心の分離**: ViewはUI更新のみを処理、Presenterはロジックを統括、Modelはデータを管理
* **ライフサイクル管理**: 統一されたライフサイクルインターフェースがUnityのMonoBehaviourコールバックを抽象化
* **フレームワーク依存なし**: 依存性注入フレームワークではなく手動のコンストラクタインジェクション

このパターンは、すべての主要なシーン（タイトル、楽曲選択、ゲームプレイ、リザルト）で一貫して適用されています。

---

## 基本クラス階層

### LifetimeScope 

`LifetimeScope`は、各シーンのエントリーポイントとして機能する抽象的な`MonoBehaviour`です。Unityのライフサイクルメソッドをプレゼンターのインターフェースベースのライフサイクルに橋渡しします。

| メソッド | Unityコールバック | Presenterインターフェース | 目的 |
| --- | --- | --- | --- |
| `Start()` | Unity Start | `IStartable.Start()` | シーン初期化 |
| `Update()` | Unity Update | `ITickable.Tick()` | フレーム毎の更新 |
| `OnDestroy()` | Unity OnDestroy | `IDisposable.Dispose()` | クリーンアップと破棄 |

**主な特徴**:

* protectedの`PresenterBase _presenter`フィールドを保持
* 派生クラスは`Awake()`をオーバーライドしてpresenterを構築
* Unityライフサイクルメソッドは拡張性のため`protected virtual`とマーク

### PresenterBase 

`PresenterBase<V, M>`は、すべてのpresenterのための汎用基本クラスです。型制約を強制し、ViewとModelへの参照を管理します。

```
PresenterBase<V, M>
  where V : ViewBase
  where M : new()
```

**責任**:

* `_view`と`_model`への不変参照を保持
* パラメーターなしコンストラクタを使用してModelをインスタンス化: `_model = new M()`
* ライフサイクルインターフェースを実装: `IStartable`、`ITickable`、`IDisposable`

**コンストラクタパターン**:

```c#
protected PresenterBase(V view)
{    
    _view = (V)view;    
    _model = new M();
}
```

Modelは自動的にインスタンス化されますが、ViewはLifetimeScopeから注入されます。この非対称性は、ViewがInspectorで設定されるUnity GameObjectであるのに対し、Modelは単純なC#オブジェクトであるためです。

### ViewBase 

`ViewBase`は、すべてのView実装の基礎となる抽象`MonoBehaviour`です。

**必須メソッド**:

* `Init()`: 初期化中にPresenterによって呼び出され、UI状態をセットアップ
* `CallUpdate()`: Presenterの`Tick()`メソッドから呼び出され、フレーム毎のUI更新を行う

Viewは、プロパティ（例: `Button.ButtonClickedEvent`）を通じてUIイベントを公開し、Presenterが表示状態を更新するためのメソッドを提供します。

---

## ライフサイクルフロー 

### UnityからPresenterへのライフサイクルマッピング 

```mermaid
sequenceDiagram
  participant p1 as Unity
  participant p2 as LifetimeScope
  participant p3 as Presenter
  participant p4 as View
  participant p5 as Model

  note over p1,p5: シーン読込
  p1->>p2: Awake()
  p2->>p3: new XxxPresenter(view)
  p3->>p5: new M()
  note over p1,p5: 初期化フェーズ
  p1->>p2: Start()
  p2->>p3: Start()
  p3->>p5: Init() / データセットアップ
  p3->>p4: Init()
  p3->>p4: SetData(...)
  note over p1,p5: 更新ループ
  loop 毎フレーム
    p1->>p2: Update()
    p2->>p3: Tick()
    p3->>p4: CallUpdate()
  end
  note over p1,p5: クリーンアップフェーズ
  p1->>p2: OnDestroy()
  p2->>p3: Dispose()
  p3->>p3: リソースをクリーンアップ
```
### ライフサイクルフェーズの詳細 

| フェーズ | アクション | 典型的な操作 |
| --- | --- | --- |
| **構築** | LifetimeScopeの`Awake()` | Presenterを作成、View参照を注入、Modelをインスタンス化 |
| **初期化** | `Start()` → `Presenter.Start()` | 保存データを読込、View状態を初期化、イベントリスナーをセットアップ |
| **実行** | `Update()` → `Presenter.Tick()` | フレーム毎のロジックを処理、入力をポーリング、アニメーションを更新 |
| **破棄** | `OnDestroy()` → `Presenter.Dispose()` | イベントを解除、リソースを解放、非同期操作をキャンセル |

このパターンは、直接的なUnity依存関係を回避することで、presenterがテスト可能であることを保証します。`IStartable`、`ITickable`、`IDisposable`インターフェースは、モックまたはテストハーネスによって実装できます。

---

## コンストラクタインジェクションパターン 

### 手動の依存性注入 

このコードベースは、ZenjectやVContainerなどのDIフレームワークではなく、**手動のコンストラクタインジェクション**を使用します。このアプローチは明示的で追跡可能ですが、各シーンのLifetimeScopeでの手動配線が必要です。

### タイトルシーンの例 

```mermaid
flowchart TD

subgraph TitleLifetimeScopeAwake ["TitleLifetimeScope (Awake)"]
    TLS["TitleLifetimeScope"]
    TSV_REF["_view (SerializeField)"]
end

subgraph シーン階層 ["シーン階層"]
    TSV["TitleSceneView (MonoBehaviour)"]
    MENU["MenuPanelController"]
    UI["UIコンポーネント (Buttons, Text)"]
end

subgraph Presenterの構築 ["Presenterの構築"]
    TSP["new TitleScenePresenter(_view)"]
    TSM["new TitleSceneModel()"]
end

TSV -->|"Inspectorで割当"| TSV_REF
TSV --> MENU
TSV --> UI
TLS -->|"Awake()"| TSP
TSP -->|"インスタンス化"| TSM
```

**実装**:

```c#
public sealed class TitleLifetimeScope : LifetimeScope
{    
    [SerializeField] TitleSceneView _view;

    void Awake()    
    {        
        _presenter = new TitleScenePresenter(_view); 
    }}
```



```c#
public sealed class TitleScenePresenter : PresenterBase<TitleSceneView, TitleSceneModel>
{
    public TitleScenePresenter(TitleSceneView view) : base(view)    
    {    
    }

    public override void Start()
    {
        Init();    
    }    
        
    // ...
}
```

**主な特徴**:

* View参照はUnity InspectorからLifetimeScopeの`[SerializeField]`に割り当て
* Presenterは`Awake()`でViewを注入して作成
* Modelは`PresenterBase<V, M>`コンストラクタによって自動的にインスタンス化
* リフレクションやサービスロケーターパターンなし

---

## シーン実装例 

### タイトルシーンのMVPトライアド 

**目的**: アプリケーション起動、ディレクトリ作成、楽曲選択への遷移を処理する最小限のシーン。

#### コンポーネントの責任

| コンポーネント | 責任 |
| --- | --- |
| **TitleSceneView** | UIアニメーション（フェードイン、バウンス）、ボタン状態、タップエフェクト |
| **TitleScenePresenter** | ディレクトリ作成、シーン遷移ロジック、エラー処理 |
| **TitleSceneModel** | `GameManager`シングルトンによって暗黙的に管理（専用のModelクラスなし） |

**Viewインターフェース**:


* プロパティ: `StartButton`、`MenuButton`
* メソッド: `Init()`、`SetVersionText()`、`AnimateTitleAsync()`、`PlayStartAnimationAsync()`

**Presenterフロー**:


1. 外部ディレクトリを作成（`ExternalDirectory.CreateSongsDirectory()`など）
2. ボタンイベントリスナーをセットアップ
3. タイトル画面をアニメーション
4. スタートボタンクリック時: 楽曲を確認、エラーダイアログを表示、またはSelectMusicシーンに遷移

---

### リザルトシーンのMVPトライアド 

**目的**: ゲームプレイ結果を表示、スコアを保存、パフォーマンスグラフを描画、ランキングを送信。

#### コンポーネントの責任

```mermaid
flowchart TD
    subgraph ResultScenePresenter ["ResultScenePresenter"]
        P_INIT["InitializeModel()<br/>InitializeView()<br/>InitializeControllers()"]
        P_SEQ["RunResultSequenceAsync()"]
        P_LOGIC["SendRankingAsync()<br/>FinishSceneAsync()"]
    end

    subgraph ResultSceneView ["ResultSceneView"]
        V_UI["ShowMainPanel()<br/>ShowJudge()<br/>ShowRank()"]
        V_PANELS["SetJacketTexture()<br/>SetMusicInfo()<br/>ShowEarlyLate()"]
        V_ANIM["PlayScoreAnimation()"]
    end

    subgraph ResultSceneModel ["ResultSceneModel"]
        M_DATA["ResultData<br/>SongInfo<br/>BeforeRecord"]
        M_CALC["GetJudges()<br/>GetMaxDigitJudge()"]
        M_SAVE["SaveScore()"]
        M_DB["ScoreDataBaseConnector"]
    end

    subgraph 追加コントローラー ["追加コントローラー"]
        GRAPH["ResultGraph"]
        TIMING["ResultAutoTimingPanel"]
        BGM["ResultBgmController"]
    end

    P_INIT -->|"データ取得"| M_DATA
    P_INIT -->|"UI初期化"| V_UI
    P_INIT -->|"コントローラー初期化"| GRAPH
    P_INIT -->|"コントローラー初期化"| TIMING
    P_INIT -->|"コントローラー初期化"| BGM
    P_SEQ -->|"表示更新"| V_PANELS
    P_SEQ -->|"アニメーション"| V_ANIM
    P_LOGIC -->|"スコア保存"| M_SAVE
    M_SAVE -->|"DB書込"| M_DB
    M_CALC -->|"判定データ"| V_UI
```

#### Model: ResultSceneModel

結果データとスコア永続化を管理します。

**主要フィールド**:

* `_resultData`: `GameManager.Instance.ResultData`からの現在のプレイ結果
* `_beforeRecord`: `ScoreDataPrefas`からの以前のハイスコア
* `_scoreDataBaseConnector`: スコア履歴のためのデータベースインターフェース

**パブリックインターフェース**:


```c#
public ResultData ResultData => _resultData;
public ScoreData BeforeRecord => _beforeRecord;
public PlayMode PlayMode => _playMode;
public SongInfo SongInfo => _selectSongInfo;
public bool IsCustomJudge => GameManager.Instance.JudgeTimeOption.IsCustom;
public bool CanSaveCustomJudgent => _canSaveCustomJudgent;
```

**主要メソッド**:

* `Init()`: GameManagerから結果データを読込
* `SaveScore()`: スコアを`ScoreDataPrefas`とLiteDBに永続化
* `GetPlayerScores()`、`GetRivalScores()`: 表示用のスコア履歴を取得
* `CanRanking()`: 判定設定に基づいてランキング適格性を検証

#### View: ResultSceneView

UIコンポーネントと更新メソッドを公開します。

**シリアライズされたコントローラー**:


* `ResultAnimationController`: パネルのフェードインアニメーションを統括
* `ClearStateViewController`: CLEAR/FULL COMBO/ALL BRILLIANTステータスを表示
* `ResultOnlinePanel`: オンラインスコア送信UI
* `ResultTimingHistogramController`: タイミング分布グラフを表示


**表示メソッド**:

* `SetMusicInfo()`: タイトル、アーティスト、難易度表示
* `ShowJudge()`: ゼロ埋めでの判定カウント表示
* `ShowEarlyLate()`: Early/Late タイミング統計
* `ShowRank()`: アニメーション付きのランク画像
* `PlayScoreAnimation()`: スコアカウントアップアニメーション

#### Presenter: ResultScenePresenter

結果表示シーケンスを統括し、ビジネスロジックを処理します。

**コンストラクタ経由で注入される依存関係**:


```c#
readonly ResultGraph _resultGraph;
readonly ResultAutoTimingPanel _resultAutoTimingPanel;
readonly ResultBgmController _resultBgmController;

public ResultScenePresenter(    
    ResultSceneView view,    
    ResultGraph resultGraph,    
    ResultAutoTimingPanel resultAutoTimingPanel,    
    ResultBgmController resultBgmController) : base(view)
```

**コールバックから非同期へのパターン**:

Presenterは、Viewのコールバックベースのメソッドを`UniTask`ベースの非同期メソッドに変換して、線形制御フローを実現します

このパターンは、コールバックのネストを排除し、シーケンシャルな非同期ロジックを可能にします。

#### LifetimeScope: ResultLifetimeScope

依存関係を配線し、presenterを構築します。

このLifetimeScopeは、複数依存関係の注入パターンを示しています：すべての依存関係はInspectorで割り当てられた`[SerializeField]`参照であり、次にPresenterコンストラクタに渡されます。

---

## サブコンポーネントのMVPパターン 

シーン内の一部のUIパネルは、完全なLifetimeScope統合なしで、類似のMVP風のパターンに従います。

### パネルコントローラー 

パネルコントローラーは、UIサブセクションのミニpresenterとして機能します：

**例**:

* `MenuPanelController`: タイトルシーンの設定メニュー 
* `ResultOnlinePanel`: オンラインランキング送信パネル 
* `ResultTimingHistogramController`: ヒストグラム表示ロジック 
* `FolderSelectPanelView`: フォルダ選択UI 

これらのコンポーネントは：

* 多くの場合、`[SerializeField]`参照を持つ`MonoBehaviour`
* `Init()`、`SetActive()`、`SetData()`のようなメソッドを公開
* 独自の内部状態を処理するが、イベントまたはコールバックを介してアクションを報告
* シーンのメインViewまたはPresenterによって参照される

### FancyScrollView統合 

楽曲選択システムは、MVPパターンに適合するように改造されたサードパーティの`FancyScrollView`ライブラリを使用します：

| コンポーネント | 役割 | ファイル |
| --- | --- | --- |
| `ScrollView` | カスタムFancyScrollView実装 |  |
| `Cell` | 個別のリストアイテムビュー |  |
| `Context` | 共有状態（選択されたインデックス、コールバック） |  |

`ScrollView`は`OnChangeSelection`などのイベントと`UpdateData()`などのメソッドを公開し、シーンのpresenterが専門的なviewコンポーネントとして扱うことができます。

---

## 主要な設計決定 

### なぜ手動コンストラクタインジェクションなのか？ 

**利点**:

* 明示的で追跡可能な依存関係グラフ
* 隠れたサービスロケーターやリフレクションがない
* 最小限の外部依存関係
* デバッグが容易（明確な構築スタックトレース）

**トレードオフ**:

* LifetimeScopeクラスでより多くのボイラープレート
* 自動ライフサイクル管理やスコープ設定がない
* 新しい依存関係を追加する際のリファクタリングが難しい

### なぜジェネリックPresenterBaseなのか？ 

ジェネリック`PresenterBase<V, M>`は以下を保証します：

* 型安全性：`_view`と`_model`は強く型付けされる
* 強制された制約：`V : ViewBase`、`M : new()`
* 自動Modelインスタンス化
* すべてのシーンで一貫したパターン

### View-Presenterイベントバインディング 

Viewは、Unityボタンイベントを直接公開します：

```c#
public Button.ButtonClickedEvent OnClickBackSelectMusicButton => _backSelectMusicButton.onClick;
```

Presenterは初期化時に購読します：

```c#
_view.OnClickBackSelectMusicButton.AddListener(() => { ... });
```

このアプローチは：

* 中間イベントラッパークラスを回避
* Unityの組み込み`UnityEvent`システムを活用
* Viewをパッシブに保つ（コールバック内にロジックなし）

---

## サマリーテーブル 

| 側面 | 実装 |
| --- | --- |
| **エントリーポイント** | `LifetimeScope` MonoBehaviour |
| **View** | `Init()`と`CallUpdate()`を持つ`ViewBase` MonoBehaviour |
| **Presenter** | `Start()`、`Tick()`、`Dispose()`を持つ`PresenterBase<V, M>` |
| **Model** | 単純なC#クラス、`new M()`でインスタンス化 |
| **依存性注入** | `LifetimeScope.Awake()`での手動コンストラクタインジェクション |
| **ライフサイクルマッピング** | Unity `Start`/`Update`/`OnDestroy` → `IStartable`/`ITickable`/`IDisposable` |
| **非同期操作** | キャンセル用の`CancellationToken`を持つ`UniTask` |
| **シーン例** | タイトル、リザルト（ここに記載）；MusicSelect、Game（[シーンフロー](#2.1)参照） |

### On this page

* [MVPパターンの実装](#2.2-mvp)
* [目的と範囲](#2.2-)
* [パターン概要](#2.2--1)
* [基本クラス階層](#2.2--2)
* [クラス構造図](#2.2--3)
* [LifetimeScope](#2.2-lifetimescope)
* [PresenterBase](#2.2-presenterbase)
* [ViewBase](#2.2-viewbase)
* [ライフサイクルフロー](#2.2--4)
* [UnityからPresenterへのライフサイクルマッピング](#2.2-unitypresenter)
* [ライフサイクルフェーズの詳細](#2.2--5)
* [コンストラクタインジェクションパターン](#2.2--6)
* [手動の依存性注入](#2.2--7)
* [タイトルシーンの例](#2.2--8)
* [シーン実装例](#2.2--9)
* [タイトルシーンのMVPトライアド](#2.2-mvp-1)
* [リザルトシーンのMVPトライアド](#2.2-mvp-2)
* [サブコンポーネントのMVPパターン](#2.2-mvp-3)
* [パネルコントローラー](#2.2--10)
* [FancyScrollView統合](#2.2-fancyscrollview)
* [主要な設計決定](#2.2--11)
* [なぜ手動コンストラクタインジェクションなのか？](#2.2--12)
* [なぜジェネリックPresenterBaseなのか？](#2.2-presenterbase-1)
* [View-Presenterイベントバインディング](#2.2-view-presenter)
* [サマリーテーブル](#2.2--13)

