

# アクターシステム 

アクターシステムは、Luaスクリプトがゲームプレイ中にゲームオブジェクトを動的に作成・制御するためのコンポーネントベースアーキテクチャを提供します。アクターは、個別にスクリプト可能なエンティティ（オーディオソース、ビデオプレイヤー、2Dスプライト、UIテキスト）を表し、ゲームプレイイベントに応答し、独立したライフサイクルを持つことができます。Lua統合アーキテクチャとスクリプト読込に関する詳細情報は、[Lua統合アーキテクチャ](#7.1)を参照してください。スクリプトで利用可能なLua APIメソッドの詳細については、[Lua APIリファレンス](#7.3)を参照してください。

---

## システム概要 

アクターシステムにより、譜面作成者は譜面のメインLuaスクリプト内から複数の独立した「アクター」をインスタンス化することで、カスタムスクリプト動作でゲームプレイを拡張できます。各アクタータイプは特定のUnityコンポーネント（AudioSource、VideoPlayer、SpriteRenderer、TextMeshPro）をラップし、ライフサイクル管理とイベント処理のための一貫したインターフェースを提供します。

### 主要な特徴 

| 側面 | 説明 |
| --- | --- |
| **目的** | Luaスクリプトで制御されるゲームオブジェクトの実行時生成を可能にする |
| **アクタータイプ** | `Actor2D`, `ActorAudio`, `ActorUIText`, `ActorVideoCanvas` |
| **ライフサイクル** | `onloaded` → `start` → `update` (フレーム毎) → `finish` → `ondestroy` |
| **イベントシステム** | アクターはゲームプレイイベント（ノーツヒット、入力、一時停止/再開）を受信 |
| **管理** | `ActorFactory`がアクターを作成; `LuaManager`がライフサイクルを管理 |
| **スクリプト分離** | 各アクターは独立した環境を持つ独自のLuaスクリプトを持つことができる |

---

## アクタータイプの階層構造 

```mermaid
flowchart TD

Actor["Actor (抽象基底クラス)"]
Actor2D["Actor2D 2Dスプライト/グラフィックス"]
ActorAudio["ActorAudio オーディオ再生"]
ActorUIText["ActorUIText UIテキスト表示"]
ActorVideoCanvas["ActorVideoCanvas ビデオ再生"]
UnityAudioSource["Unity AudioSource"]
UnityVideoPlayer["Unity VideoPlayer + Canvas + RawImage"]
UnitySpriteRenderer["Unity SpriteRenderer"]
UnityTMP["Unity TextMeshPro"]
LuaScript["Luaスクリプト (オプション)"]

Actor -.->|"継承"| Actor2D
Actor -.->|"制御"| ActorAudio
Actor -.->|"継承"| ActorUIText
Actor -.->|"継承"| ActorVideoCanvas
ActorAudio -.->|"ラップ"| UnityAudioSource
ActorVideoCanvas -.->|"ラップ"| UnityVideoPlayer
Actor2D -.->|"ラップ"| UnitySpriteRenderer
ActorUIText -.->|"ラップ"| UnityTMP
LuaScript -.->|"継承"| Actor2D
LuaScript -.->|"制御"| ActorAudio
LuaScript -.->|"制御"| ActorUIText
LuaScript -.->|"制御"| ActorVideoCanvas
```

**アクタータイプの機能**

## Actor基底クラス 

`Actor`抽象クラスは、すべてのアクタータイプの基盤を提供し、ライフサイクル管理、Luaスクリプト統合、イベント処理を実装します。

### コア構造 

```mermaid
flowchart TD

ScriptEnv["LuaTable _scriptEnv スクリプト環境"]
Callbacks["ライフサイクルデリゲート _luaStart _luaUpdate _luaFinish _luaOnDestroy"]
Events["イベントデリゲート _luaOnHitNote _luaOnMissedNote _luaOnSpawnNote _luaOnPause _luaOnResume _luaOnGameOver _luaOnInput..."]
Cache["関数キャッシュ _actionCache _funcDelegateCache"]

subgraph Actorコンポーネント ["Actorコンポーネント"]
    ScriptEnv
    Callbacks
    Events
    Cache
    ScriptEnv -.->|"コールバックを抽出"| Callbacks
    ScriptEnv -.->|"イベントを抽出"| Events
    Cache -.->|"繰り返し呼び出しを最適化"| ScriptEnv
end
```

 

### ライフサイクルメソッド 

| メソッド | トリガー | 目的 |
| --- | --- | --- |
| `Init(name, scriptEnv, onError, actorTypeName)` | アクター作成時 | スクリプト環境を初期化し、Luaコールバックを抽出 |
| `LuaStart()` | ゲーム開始時 | Lua `start()` 関数を呼び出し |
| `LuaUpdate()` | 毎フレーム | アクターがアクティブな場合、Lua `update()` 関数を呼び出し |
| `LuaFinish()` | ゲーム終了時 | Lua `finish()` 関数を呼び出し |
| `OnDestroy()` | アクター破棄時 | Lua `ondestroy()` を呼び出し、リソースをクリーンアップ |

### イベント処理メソッド 

`Actor`基底クラスは、以下のメソッドを通じてゲームプレイイベントをLuaスクリプトに転送します:

```mermaid
flowchart TD

OnHitNote["LuaOnHitNote(id, lane, noteType, judge, isAttack)"]
OnMissedNote["LuaOnMissedNote(id, lane, noteType)"]
OnSpawnNote["LuaOnCreateNote(noteController)"]
OnSpawnLong["LuaOnCreateLong(longController)"]
OnPause["LuaOnPause()"]
OnResume["LuaOnResume()"]
OnGameOver["LuaOnGameOver()"]
OnInputDown["LuaOnInputDown(touchId, posX, screenPosX, screenPosY)"]
OnInputMove["LuaOnInputMove(...)"]
OnInputUp["LuaOnInputUp(...)"]
LuaScript["Luaスクリプト関数"]

OnHitNote -.->|"呼び出し"| LuaScript
OnMissedNote -.->|"呼び出し"| LuaScript
OnSpawnNote -.->|"呼び出し"| LuaScript
OnSpawnLong -.->|"呼び出し"| LuaScript
OnPause -.->|"呼び出し"| LuaScript
OnResume -.->|"呼び出し"| LuaScript
OnGameOver -.->|"呼び出し"| LuaScript
OnInputDown -.->|"呼び出し"| LuaScript
OnInputMove -.-> LuaScript
OnInputUp -.-> LuaScript

subgraph 入力イベント ["入力イベント"]
    OnInputDown
    OnInputMove
    OnInputUp
end

subgraph システムイベント ["システムイベント"]
    OnPause
    OnResume
    OnGameOver
end

subgraph ゲームプレイイベント ["ゲームプレイイベント"]
    OnHitNote
    OnMissedNote
    OnSpawnNote
    OnSpawnLong
end
```

 

### 関数キャッシング 

繰り返されるLua関数呼び出しを最適化するため、`Actor`クラスは2つのキャッシングメカニズムを実装しています:

```mermaid
flowchart TD

InvokeFunctionBase["InvokeFunctionBase(functionName)"]
ActionCache["_actionCache Dictionary"]
FuncDelegateCache["_funcDelegateCache Dictionary"]
ScriptEnv["_scriptEnv.Get(functionName)"]
LuaFunction["Lua関数"]
InvokeFunctionBaseInt["InvokeFunctionBase(functionName, int)"]

InvokeFunctionBase -.->|"初回呼び出し"| ScriptEnv
ScriptEnv -.->|"デリゲートを保存"| ActionCache
InvokeFunctionBase -.->|"以降の呼び出し"| ActionCache
ActionCache -.->|"高速実行"| LuaFunction
InvokeFunctionBaseInt -.->|"パラメータ付き"| FuncDelegateCache
```

 

### エラーハンドリング 

Luaスクリプトが実行中に例外をスローした場合、`Actor`クラスは:

1. `_onError`コールバックを呼び出す（通常はゲームを一時停止）
2. アクター名とエラーメッセージを含むエラーダイアログを表示
3. 楽曲選択シーンに戻るボタンを提供
4. 必要に応じて背景リソースをクリーンアップ

 

---

## 具体的なアクタータイプ 

### ActorAudio 

Luaスクリプトで制御されるオーディオ再生のためにUnity `AudioSource`コンポーネントをラップします。

**主要機能:**

* 譜面フォルダからの非同期オーディオ読込: `LoadAudio(filePath, onComplete)`
* プリセットメソッドによる音量制御: `SetSystemSEVolume()`, `SetTouchSEVolume()`, `SetMusicVolume()`
* 再生制御: `Play()`, `Stop()`, `Pause()`
* `GetAudioSource()`を介した`AudioSource`コンポーネントへの直接アクセス

**実装詳細:**

| メソッド | 説明 |
| --- | --- |
| `LoadAudioAsync(ct, filePath, onComplete)` | `AudioClipLoader`を使用して`file://`パスからオーディオを読込 |
| `SetVolume(volume)` | `Mathf.Clamp01`を使用して0-1の範囲で音量を設定 |
| `OnDestroy()` | 保留中の非同期操作をキャンセル |

 

### ActorVideoCanvas 

ビデオ再生のためにUnity `VideoPlayer`を`Canvas`、`CanvasScaler`、`RawImage`と共にラップします。

**主要機能:**

* ビデオパス設定: `SetVideoPath(filePath)`は`VideoPlayer.Prepare()`をトリガー
* `VideoTimeReference.ExternalTime`による自動音楽同期
* 画面マッチモード: `Expand`（内側にフィット）または`Shrink`（外側にフィット）
* 視覚効果のための明るさとアルファ制御
* オフセット時間調整: `SetOffsetTime(offsetTime)`で同期補正

**同期メカニズム:**

```mermaid
sequenceDiagram
  participant p1 as LuaManager
  participant p2 as ActorVideoCanvas
  participant p3 as VideoPlayer
  participant p4 as MusicManager

  p1->>p2: CallUpdate()
  p2->>p4: GetMusicTime()
  p4-->>p2: currentTime
  p2->>p2: _offsetTimeを追加
  p2->>p3: externalReferenceTimeを設定
  note over p2: 自動同期は SetAutoMusicSync(false)で無効化可能
```

**画面サイズ調整:**

アクターは画面のアスペクト比に基づいて`RawImage`サイズを動的に調整します:

* **SetInSize (Expandモード)**: ビデオは画面境界内にフィット、レターボックスが発生する可能性あり
* **SetOutSize (Shrinkモード)**: ビデオは画面全体をカバー、トリミングされる可能性あり

 

---

## ActorFactory 

`ActorFactory`クラスは、アクターのインスタンス化とLuaスクリプト統合を処理します。

### ファクトリーパターンの実装 

```mermaid
flowchart TD

Create2D["Create2D(luaFileName, hasInjection)"]
CreateAudio["CreateAudio(luaFileName)"]
CreateUIText["CreateUIText(luaFileName)"]
CreateVideoCanvas["CreateVideoCanvas(luaFileName)"]
Prefab2D["_actor2DPrefab"]
PrefabAudio["_actorAudioPrefab"]
PrefabUIText["_actorUITextPrefab"]
PrefabVideo["_actorVideoCanvasPrefab"]
ReadLua["ReadLua(luaFileName)"]
CreateScriptEnv["LuaManager.CreateScriptEnv()"]
DoString["LuaEnv.DoString(luaString)"]
ActorList["LuaManager.ActorList.Add()"]
VideoList["LuaManager.ActorVideoCanvasList.Add()"]
LuaTable["LuaTable scriptEnv"]

Create2D -.->|"luaFileName提供時"| Prefab2D
CreateAudio -.->|"インスタンス化"| PrefabAudio
CreateUIText -.->|"インスタンス化"| PrefabUIText
CreateVideoCanvas -.->|"インスタンス化"| PrefabVideo
Create2D -.->|"Actor.Init()"| ReadLua
DoString -.-> LuaTable
Create2D -.-> LuaTable
Create2D -.-> ActorList
CreateVideoCanvas -.-> VideoList

subgraph 登録 ["登録"]
    ActorList
    VideoList
end

subgraph スクリプト読込 ["スクリプト読込"]
    ReadLua
    CreateScriptEnv
    DoString
    ReadLua -.->|"インスタンス化"| CreateScriptEnv
    CreateScriptEnv -.->|"返す"| DoString
end

subgraph プレハブのインスタンス化 ["プレハブのインスタンス化"]
    Prefab2D
    PrefabAudio
    PrefabUIText
    PrefabVideo
end

subgraph ActorFactory ["ActorFactory"]
    Create2D
    CreateAudio
    CreateUIText
    CreateVideoCanvas
end
```

 

### 作成フローの詳細 

各`Create*`メソッドは以下のパターンに従います:

1. **プレハブのインスタンス化**: `GameObject.Instantiate(prefab)`でUnity GameObjectを作成
2. **Luaスクリプトの読込** (オプション): `luaFileName`が提供された場合、`ReadLua(luaFileName)`を呼び出し
3. **スクリプト環境の作成**: `LuaManager.CreateScriptEnv()`で分離されたLuaテーブルを作成
4. **Luaコードの実行**: `LuaEnv.DoString(luaString, "Lua", scriptEnv)`でスクリプトを実行
5. **self参照の注入**: `scriptEnv.Set("self", actor)`でアクターをLuaからアクセス可能にする
6. **アクターの初期化**: `actor.Init(luaFileName, scriptEnv, _pause, actorTypeName)`
7. **マネージャーに登録**: `LuaManager.ActorList`に追加（必要に応じて特殊リストにも追加）

 

 

### 親Transformの割り当て 

異なるアクタータイプは異なるTransformに親として設定されます:

| アクタータイプ | 親Transform | 目的 |
| --- | --- | --- |
| `Actor2D` | ワールドルート | 3D空間での位置決め |
| `ActorAudio` | ワールドルート | 3Dオーディオソース |
| `ActorUIText` | `_luaOverrayPanel` | UIオーバーレイ描画 |
| `ActorVideoCanvas` | ワールドルート | スクリーン空間キャンバス |

 

---

## LuaManagerとの統合 

`LuaManager`はすべてのアクターのライフサイクルイベントを調整し、アクターコレクションを維持します。

### アクターコレクション 

```mermaid
flowchart TD

LuaManager["LuaManager"]
ActorList["List _actorList"]
VideoList["List _actorVideoCanvasList"]
Actor2D["Actor2Dインスタンス"]
ActorAudio["ActorAudioインスタンス"]
ActorUIText["ActorUITextインスタンス"]
ActorVideoCanvas["ActorVideoCanvasインスタンス"]

LuaManager -.->|"特殊ストレージ"| ActorList
LuaManager -.->|"含む"| VideoList
Actor2D -.->|"格納"| ActorList
ActorAudio -.->|"格納"| ActorList
ActorUIText -.->|"両方に格納"| ActorList
ActorVideoCanvas -.->|"含む"| ActorList
ActorVideoCanvas -.-> VideoList

subgraph アクタータイプ ["アクタータイプ"]
    Actor2D
    ActorAudio
    ActorUIText
    ActorVideoCanvas
end

subgraph アクターストレージ ["アクターストレージ"]
    ActorList
    VideoList
end
```

**注意**: `ActorVideoCanvas`インスタンスは両方のリストに格納されます。これは、標準的なライフサイクル管理に加えて、特殊な更新処理（ビデオ同期）が必要なためです。

 

 

### イベント配信パターン 

`LuaManager`は、イベント処理中に破棄される可能性のあるアクターを安全に処理するため、逆順イテレーションパターンを使用してすべてのアクターにイベントを配信します:

```mermaid
flowchart TD

GameEvent["ゲームイベント (例: OnHitNote)"]
LuaManager["LuaManager.OnHitNote()"]
MainScript["メインLuaスクリプト _luaOnHitNote()"]
ActorLoop["アクターをイテレート (逆順)"]
ActorCheck["HasLuaOnHitNoteをチェック"]
ActorInvoke["actor.LuaOnHitNote()"]
ErrorHandler["エラーハンドリング"]

GameEvent -.-> LuaManager
LuaManager -.-> MainScript
LuaManager -.-> ActorLoop
ActorLoop -.->|"trueの場合"| ActorCheck
ActorCheck -.->|"try-catch"| ActorInvoke
ActorInvoke -.-> ErrorHandler
```

**コードパターン** (各イベントタイプで繰り返し):

```
int loop = _actorList.Count - 1;
if (loop >= 0)
{
    for (int i = loop; i >= 0; i--)
    {
        if (_actorList[i] is not null && _actorList[i].HasLuaOnHitNote)
        {
            _actorList[i].LuaOnHitNote(id, lane, noteType, judge, isAttack);
        }
    }
}
```
 

## 完全なライフサイクルとイベントフロー 

### アクターの作成と初期化 

```mermaid
flowchart TD

Start["譜面Luaスクリプト実行"]
CallFactory["ACTORFACTORY:Create2D('script.lua')"]
Instantiate["ActorFactoryがプレハブをインスタンス化"]
LoadScript["ActorFactory.ReadLua('script.lua')"]
CreateEnv["LuaManager.CreateScriptEnv()"]
ExecuteLua["LuaEnv.DoString(luaString, scriptEnv)"]
ExtractCallbacks["Actor.Init()がコールバックを抽出"]
RegisterActor["LuaManager.ActorListに追加"]
OnLoaded["存在する場合、Lua onloaded()を実行"]
Ready["アクター準備完了"]
Callbacks["start update finish ondestroy onHitNote onMissedNote ..."]

Start -.-> CallFactory
CallFactory -.-> Instantiate
Instantiate -.-> LoadScript
LoadScript -.-> CreateEnv
CreateEnv -.->|"scriptEnv.Get()"| ExecuteLua
ExecuteLua -.-> ExtractCallbacks
ExtractCallbacks -.-> RegisterActor
RegisterActor -.-> OnLoaded
OnLoaded -.-> Ready
ExtractCallbacks -.-> Callbacks
```

 

 

### ゲームプレイイベントフロー 

ゲームプレイ中にノーツがヒットされた場合:

```mermaid
sequenceDiagram
  participant p1 as JudgeManager
  participant p2 as MusicGame
  participant p3 as LuaManager
  participant p4 as MainScript
  participant p5 as ActorList
  participant p6 as Actor1
  participant p7 as Actor2

  p1->>p2: OnAfterJudgeイベント
  p2->>p3: OnHitNote(id | lane | noteType | judge | isAttack)
  alt メインスクリプトがonHitNoteを持つ場-
    p3->>p4: _luaOnHitNote(...)
    p4-->>p4: カスタムロジックを実行
  end
  p3->>p5: イテレート (逆順)
  alt Actor1がonHitNo-
    p5->>p6: LuaOnHitNote(...)
    p6->>p6: Lua関数を実行
  end
  alt Actor2がonHitNoteを持つ場合
    p5->>p7: LuaOnHitNote(...)
    p7->>p7: Lua関数を実行
  end
  note over p3: 各呼び出しはtry-catchでラップ エラーは一時停止とダイアログをトリガー
```

 

---

## スクリプト環境とLua統合 

### スクリプト環境の分離 

各アクターは、グローバル関数と注入された変数にアクセスできる独自の分離されたLua環境（LuaTable）を持つことができます:

```mermaid
flowchart TD

GlobalEnv["LuaEnv.Global (共有環境)"]
MainEnv["LuaTable _scriptEnv (メイン)"]
MainMeta["__index = Global"]
Actor1Env["LuaTable scriptEnv (Actor1)"]
Actor1Meta["__index = Global"]
Actor1Self["self = Actor1インスタンス"]
Actor2Env["LuaTable scriptEnv (Actor2)"]
Actor2Meta["__index = Global"]
Actor2Self["self = Actor2インスタンス"]
GlobalVars["GAMESTATE SCREENMAN ACTORFACTORY SONGMAN UTIL など"]

GlobalEnv -.->|"注入"| MainMeta
GlobalEnv -.->|"メタテーブル"| Actor1Meta
GlobalEnv -.->|"注入"| Actor2Meta
GlobalVars -.-> GlobalEnv

subgraph Actor2スクリプト環境 ["Actor2スクリプト環境"]
    Actor2Env
    Actor2Meta
    Actor2Self
    Actor2Meta -.->|"メタテーブル"| Actor2Env
    Actor2Self -.-> Actor2Env
end

subgraph Actor1スクリプト環境 ["Actor1スクリプト環境"]
    Actor1Env
    Actor1Meta
    Actor1Self
    Actor1Meta -.->|"すべてからアクセス可能"| Actor1Env
    Actor1Self -.-> Actor1Env
end

subgraph メインスクリプト環境 ["メインスクリプト環境"]
    MainEnv
    MainMeta
    MainMeta -.->|"メタテーブル"| MainEnv
end
```

**環境作成:**

`CreateScriptEnv()`メソッドは、グローバル環境へのメタテーブルフォールバックを持つ分離されたLuaテーブルを作成します:

```
local scriptEnv = {}
setmetatable(scriptEnv, { __index = _G })
```

これにより、各アクターのスクリプトは以下が可能になります:

* グローバルLua APIオブジェクト（GAMESTATE、SCREENMANなど）へのアクセス
* 他のスクリプトと競合しない独自のローカル変数を持つ
* `self`変数を介して自身を参照

 

 

### グローバル変数の注入 

`LuaManager`は、すべてのスクリプト（メインとアクター）がアクセスできるグローバルLua環境にC#オブジェクトを注入します:

| 変数名 | C#型 | 目的 |
| --- | --- | --- |
| `GAMESTATE` | `GameState` | ゲーム速度、距離、ノーツプロパティを制御 |
| `SCREENMAN` | `ScreenMan` | 画面情報、背景制御、メッセージ |
| `ACTORFACTORY` | `ActorFactory` | 新しいアクターを作成 |
| `SONGMAN` | `SongMan` | 楽曲メタデータ、BPM、スクロールデータにアクセス |
| `PLAYERSTATS` | `PlayerStats` | スコア、コンボ、ライフ、プレイヤーオプションにアクセス |
| `CAMERAMAN` | `CameraMan` | カメラの位置と回転を制御 |
| `UTIL` | `Util` | ユーティリティ関数（テクスチャ、トゥイーン、数学） |
| `ASSETMAN` | `AssetMan` | 外部アセットを読込 |
| `DANMAKUSTAGE` / `DSTAGE` | `DanmakuStageMan` | 弾幕パターンシステム |

 

---

## メモリ管理とクリーンアップ 

### アクターの破棄 

アクターが破棄される場合（明示的またはシーン終了時）:

1. **Luaコールバック**: 定義されている場合、`_luaOnDestroy()`が呼び出される
2. **キャッシュのクリーンアップ**: `_actionCache`と`_funcDelegateCache`がクリア
3. **デリゲートのクリーンアップ**: すべての保存されたデリゲートが`null`に設定
4. **スクリプト環境**: `_scriptEnv.Dispose()`がLuaテーブルを解放
5. **Unityクリーンアップ**: GameObjectはUnityのライフサイクルによって破棄

 

### LuaManagerのクリーンアップ 

`LuaManager`が破棄される場合:

```mermaid
flowchart TD

OnDestroy["LuaManager.OnDestroy()"]
CallScriptDestroy["_luaOnDestroy()を呼び出し"]
FinalizeDanmaku["DanmakuStageMan.OnFinalize()"]
DisposeScriptEnv["_scriptEnv.Dispose()"]
NullifyDelegates["すべてのデリゲートをnullに設定"]
ClearArrays["注入配列をクリア"]
Note["注意: 個々のアクターは Unityによって自動的に破棄 シーンアンロード時にGameObjectが破棄される"]

OnDestroy -.-> CallScriptDestroy
OnDestroy -.-> FinalizeDanmaku
OnDestroy -.-> DisposeScriptEnv
OnDestroy -.-> NullifyDelegates
OnDestroy -.-> ClearArrays
```

**重要**: `LuaManager`は明示的にアクターを破棄しません。UnityのシーンアンロードプロセスがGameObjectの破棄を処理し、各アクターの`OnDestroy()`メソッドをトリガーします。

 

### Luaガベージコレクション 

`LuaManager`はLua環境で定期的なガベージコレクションを実行します:

```
if (Time.time - g_lastGCTime > GC_INTERVAL)
{    
    g_luaEnv.Tick();    
    g_lastGCTime = Time.time;
}
```

これは`CallUpdate()`中に1秒毎（`GC_INTERVAL`で定義）に実行され、一時的なLuaオブジェクトからのメモリ蓄積を防ぎます。

 

 

---

## 使用パターン 

### メインLuaスクリプトからアクターを作成 

譜面のメインLuaスクリプトでの典型的なパターン:

```lua
-- 後で参照するためにグローバルアクターを保存
local audioActor = nil
local videoActor = nil

function onloaded()    
    -- 独自のスクリプトを持つオーディオアクターを作成
    audioActor = ACTORFACTORY:Create2D("actors/background.lua")        
    -- スクリプトなしでオーディオアクターを作成（直接制御）
    audioActor = ACTORFACTORY:CreateAudio()    
    audioActor:LoadAudio("sound/effect.wav")        
    -- 自動音楽同期でビデオを作成
    videoActor = ACTORFACTORY:CreateVideoCanvas()    
    videoActor:SetVideoPath("video/bg.mp4")    
    videoActor:SetOffsetTime(-0.5) 
    -- 音楽の0.5秒前に開始
end

function start()    
    audioActor:Play()    
    videoActor:Play()
end
```

### 独立したスクリプトを持つアクター 

アクターのLuaスクリプト（例: `actors/particle.lua`）:

```lua
-- selfはActorインスタンスを参照
local sprite = nil
local time = 0

function onloaded()
    local texture = UTIL:LoadTexture("particle.png")
    sprite = UTIL:CreateSprite(texture)    
    self:SetSprite(sprite)
end

function update()    
    time = time + 1/60    
    local x = math.sin(time) * 100    
    local y = math.cos(time) * 100    
    self:SetPosition(x, y, 0)
end

function onHitNote(id, lane, noteType, judge, isAttack)    -- ノーツヒットに反応    
    self:SetScale(1.5, 1.5, 1.5)
end
```

### スクリプトなしのアクター（直接制御） 

```lua
local audioPlayer = nil

function onloaded()
    audioPlayer = ACTORFACTORY:CreateAudio() 
    -- スクリプトなし    
    audioPlayer:LoadAudio(
        "voice/countdown.wav", 
        function()
            SCREENMAN:SystemMessage("Audio loaded!")    
        end
    )
end
function start()    
    audioPlayer:SetVolume(0.8)    
    audioPlayer:Play()
end
```

---

## エラーハンドリングとデバッグ 

 
### Try-Catchラッピング 

`Actor`クラスのすべてのLuaコールバック呼び出しは、try-catchブロックでラップされています:

```c#
try
{    
    _luaUpdate();
}
catch (Exception e)
{    
    _onError?.Invoke();  
    // ゲームを一時停止    
    ShowErrorDialog(e.Message);    
    return;
}
```

これにより、単一のアクターのエラーがゲーム全体をクラッシュさせることを防ぎながら、明確なエラー情報を提供します。

 

---

## パフォーマンスに関する考慮事項 

### 関数キャッシング 

`Actor`クラスは、繰り返しのルックアップを避けるためにLua関数参照をキャッシュします:

| キャッシュタイプ | 使用ケース | パフォーマンス影響 |
| --- | --- | --- |
| `_actionCache` | パラメータなし関数 | 呼び出し毎の辞書ルックアップを排除 |
| `_funcDelegateCache` | intパラメータを持つ関数 | 呼び出し毎の辞書ルックアップを排除 |

これは、`InvokeFunctionBase()`を介してアクター固有のメソッドから呼び出される頻繁に呼ばれる関数にとって特に重要です。

 

### 逆順イテレーション 

イベント配信ループは逆順でアクターをイテレート（`for (int i = loop; i >= 0; i--)`）します。これにより、アクターがイベント処理中に安全に自身をリストから削除でき、イテレーションインデックスに影響を与えません。

 

### ビデオアクターの分離 

`ActorVideoCanvas`インスタンスは、メインの`_actorList`に加えて、別の`_actorVideoCanvasList`に保存されます。これにより、特殊な更新ロジック（ビデオ同期）をビデオアクターのみで実行でき、他のアクタータイプでの不要なチェックを回避できます。

 

---

## まとめ 

アクターシステムは、Lua制御されたゲームオブジェクトのための柔軟なイベント駆動型アーキテクチャを提供します。主要な設計原則は以下の通りです:

* **型安全性**: 具体的なアクタータイプが特定のUnityコンポーネントをラップ
* **ライフサイクルの一貫性**: すべてのアクターが同じライフサイクルパターンに従う
* **イベントブロードキャスト**: ゲームプレイイベントが関心のあるアクターに自動的に配信
* **スクリプト分離**: 各アクターが独立したLua環境を持つことができる
* **エラー回復性**: 包括的なエラーハンドリングがカスケード障害を防ぐ
* **パフォーマンス最適化**: 関数キャッシングと特殊コレクションがオーバーヘッドを最小化

このシステムにより、譜面作成者はコアエンジンコードを変更することなく、複雑な視覚効果、カスタムオーディオキュー、インタラクティブ要素を実装できます。

### On this page

* [アクターシステム](#7.2-)
* [システム概要](#7.2--1)
* [主要な特徴](#7.2--2)
* [アクタータイプの階層構造](#7.2--3)
* [Actor基底クラス](#7.2-actor)
* [コア構造](#7.2--4)
* [ライフサイクルメソッド](#7.2--5)
* [イベント処理メソッド](#7.2--6)
* [関数キャッシング](#7.2--7)
* [エラーハンドリング](#7.2--8)
* [具体的なアクタータイプ](#7.2--9)
* [ActorAudio](#7.2-actoraudio)
* [ActorVideoCanvas](#7.2-actorvideocanvas)
* [ActorFactory](#7.2-actorfactory)
* [ファクトリーパターンの実装](#7.2--10)
* [作成フローの詳細](#7.2--11)
* [親Transformの割り当て](#7.2-transform)
* [LuaManagerとの統合](#7.2-luamanager)
* [アクターコレクション](#7.2--12)
* [イベント配信パターン](#7.2--13)
* [ライフサイクルイベントの伝播](#7.2--14)
* [完全なライフサイクルとイベントフロー](#7.2--15)
* [アクターの作成と初期化](#7.2--16)
* [ゲームプレイイベントフロー](#7.2--17)
* [スクリプト環境とLua統合](#7.2-lua)
* [スクリプト環境の分離](#7.2--18)
* [グローバル変数の注入](#7.2--19)
* [メモリ管理とクリーンアップ](#7.2--20)
* [アクターの破棄](#7.2--21)
* [LuaManagerのクリーンアップ](#7.2-luamanager-1)
* [Luaガベージコレクション](#7.2-lua-1)
* [使用パターン](#7.2--22)
* [メインLuaスクリプトからアクターを作成](#7.2-lua-2)
* [独立したスクリプトを持つアクター](#7.2--23)
* [スクリプトなしのアクター（直接制御）](#7.2--24)
* [エラーハンドリングとデバッグ](#7.2--25)
* [エラー表示メカニズム](#7.2--26)
* [Try-Catchラッピング](#7.2-try-catch)
* [パフォーマンスに関する考慮事項](#7.2--27)
* [関数キャッシング](#7.2--28)
* [逆順イテレーション](#7.2--29)
* [ビデオアクターの分離](#7.2--30)
* [まとめ](#7.2--31)

