

# アプリケーション構造 

## 目的と範囲 

本ドキュメントでは、リズムゲームの全体的なアプリケーション構造について説明します。これには、シーン構成、アーキテクチャパターン、グローバルステート管理、およびデータ永続化メカニズムが含まれます。このページでは、アプリケーションの構造と主要コンポーネントの相互作用について、高レベルの概要を提供します。

特定のサブシステムに関する詳細情報については、以下を参照してください：

* シーンの初期化と遷移：[シーンフローとライフサイクル](#2.1)を参照
* MVPパターンの詳細とプレゼンターのライフサイクル：[MVPパターンの実装](#2.2)を参照
* データのシリアル化とストレージシステム：[データ永続化システム](#2.3)を参照

---

## シーン構造 

アプリケーションは、ゲームのユーザーフローを形成する4つの主要なシーンで構成されています：

| シーン名 | Unityシーンファイル | 目的 |
| --- | --- | --- |
| **Title** | `Title.unity` | 初期エントリーポイント、メニューと設定を表示 |
| **Select Music** | `SelectMusic.unity` | 楽曲の閲覧と選択、プレイオプションの設定 |
| **Game** | `Game.unity` | ノート判定とスコアリングを含むコアゲームプレイ |
| **Result** | `Result.unity` | パフォーマンス統計とタイミングデータの表示 |

### シーンフロー図 

```mermaid
flowchart TD

AppStart["アプリケーション開始"]
Title["Titleシーン (Title.unity)"]
SelectMusic["Select Musicシーン (SelectMusic.unity)"]
Game["Gameシーン (Game.unity)"]
Result["Resultシーン (Result.unity)"]
GM["GameManager シングルトン (シーン間で永続化)"]

AppStart -.->|"管理"| Title
Title -.->|"スタートボタン"| SelectMusic
SelectMusic -.->|"楽曲選択 + プレイ"| Game
Game -.->|"楽曲完了"| Result
Result -.->|"戻る"| SelectMusic
SelectMusic -.->|"タイトルに戻る"| Title
GM -.->|"管理"| Title
GM -.->|"管理"| SelectMusic
GM -.->|"管理"| Game
GM -.-> Result
```

出典：

 

---

## MVPパターン構造 

各シーンは、`LifetimeScope`エントリーポイントを持つ**Model-View-Presenter (MVP)**パターンを実装しています。このアーキテクチャは関心事を分離します：

* **LifetimeScope**: MonoBehaviourエントリーポイント、`Awake()`中にMVPトライアドを構築
* **Presenter**: ビジネスロジックを含み、ViewとModelを調整
* **View**: UIレンダリングとユーザー入力を処理
* **Model**: （オプション）データとビジネスルールをカプセル化

### MVPクラス階層 

```mermaid
flowchart TD

LTS["LifetimeScope (抽象MonoBehaviour)"]
PRES["PresenterBase IStartable, ITickable, IDisposable"]
VIEW["ViewBase ICallUpdate"]
TLS["TitleLifetimeScope"]
TSP["TitleScenePresenter"]
TSV["TitleSceneView"]
MSLS["MusicSelectSceneLifeTimeScope"]
MSP["MusicSelectScenePresenter"]
MSV["MusicSelectSceneView"]
GLS["GameLifetimeScope"]
MGSC["MusicGameSceneController (Presenter)"]
RLS["ResultLifetimeScope"]
RSP["ResultScenePresenter"]
RSV["ResultSceneView"]

LTS -.->|"継承"| TLS
LTS -.->|"継承"| MSLS
LTS -.->|"継承"| GLS
LTS -.->|"継承"| RLS
PRES -.->|"継承"| TSP
PRES -.->|"生成"| MSP
PRES -.->|"継承"| MGSC
PRES -.->|"継承"| RSP
VIEW -.->|"継承"| TSV
VIEW -.->|"参照を渡す"| MSV
VIEW -.->|"参照を渡す"| RSV

subgraph Resultシーン ["Resultシーン"]
    RLS
    RSP
    RSV
    RLS -.->|"生成"| RSP
    RLS -.->|"参照を渡す"| RSV
    RSP -.->|"継承"| RSV
end

subgraph Gameシーン ["Gameシーン"]
    GLS
    MGSC
    GLS -.->|"生成"| MGSC
end

subgraph subGraph2 ["Select Musicシーン"]
    MSLS
    MSP
    MSV
    MSLS -.->|"制御"| MSP
    MSLS -.->|"継承"| MSV
    MSP -.->|"制御"| MSV
end

subgraph Titleシーン ["Titleシーン"]
    TLS
    TSP
    TSV
    TLS -.->|"生成"| TSP
    TLS -.->|"制御"| TSV
    TSP -.->|"継承"| TSV
end

subgraph 基底クラス ["基底クラス"]
    LTS
    PRES
    VIEW
end
```
 

### Presenterライフサイクル 

Presenterは3つのライフサイクルインターフェースを実装します：

| インターフェース | メソッド | 呼び出し元 | 目的 |
| --- | --- | --- | --- |
| `IStartable` | `Start()` | `LifetimeScope.Start()` | シーン開始時にPresenterを初期化 |
| `ITickable` | `Tick()` | `LifetimeScope.Update()` | フレーム毎の更新 |
| `IDisposable` | `Dispose()` | `LifetimeScope.OnDestroy()` | シーン破棄時のクリーンアップ |

**例：Titleシーンの初期化**

`TitleLifetimeScope`は`Awake()`中にPresenterを構築します：

```c#
// TitleLifetimeScope.cs
void Awake()
{
    _presenter = new TitleScenePresenter(_view);
}
```

出典：

 

---

## グローバルステート管理 

### GameManagerシングルトン 

`GameManager`クラスは、シーン遷移を生き延び、グローバルなアプリケーション状態を維持する永続的なシングルトンです。これは`SingletonMonoBehaviour`として実装され、`DontDestroyOnLoad`でマークされています。

```mermaid
flowchart TD

GM["GameManager"]
NOTES["NotesOption (速度、サイズ、スキン)"]
DISPLAY["DisplayOption (FPS、ディマー、エフェクト)"]
VOLUME["VolumeOption (音楽、SE、音量)"]
JUDGE["JudgeTimeOption (タイミングウィンドウ)"]
SONG["SelectSongInfo (選択された楽曲)"]
DIFF["SelectDifficulty"]
TIMING["LocalTiming"]
RESULT["ResultData"]
ONLINE["OnlinePlayerResults"]
ACCOUNT["Account"]
ONLINE_SET["OnlineSettings"]
CACHE["SongInfoCache"]
PREFAS["GameSettingsPrefas"]
ONLINE_PREFAS["OnlineSettingsPrefas"]

PREFAS -.-> NOTES
PREFAS -.-> DISPLAY
PREFAS -.-> VOLUME
PREFAS -.-> JUDGE

subgraph subGraph4 ["GameManager シングルトン"]
    GM
    GM -.->|"読込"| NOTES
    GM -.->|"読込"| DISPLAY
    GM -.->|"読込"| VOLUME
    GM -.->|"読込"| JUDGE
    GM -.->|"読込"| SONG
    GM -.-> DIFF
    GM -.-> TIMING
    GM -.-> RESULT
    GM -.-> ONLINE
    GM -.-> ACCOUNT
    GM -.-> ONLINE_SET
    GM -.-> CACHE

subgraph アカウントと設定 ["アカウントと設定"]
    ACCOUNT
    ONLINE_SET
    CACHE
end

subgraph リザルトデータ ["リザルトデータ"]
    RESULT
    ONLINE
end

subgraph 楽曲選択状態 ["楽曲選択状態"]
    SONG
    DIFF
    TIMING
end

subgraph ゲームオプション ["ゲームオプション"]
    NOTES
    DISPLAY
    VOLUME
    JUDGE
end
end
```

出典：

### GameManagerの主要な責務 

| 責務 | メソッド/プロパティ |
| --- | --- |
| **オプション管理** | `NotesOption`, `DisplayOption`, `VolumeOption`, `JudgeTimeOption` |
| **シーン遷移** | `ChangeSceneAsync()`, `ChangeScene()` |
| **楽曲選択** | `SelectSongInfo`, `SelectDifficulty`, `SelectDataIndex` |
| **リザルト保存** | `ResultData`, `OnlinePlayerResults` |
| **キャッシュ管理** | `ClearSongsCache()`, `GroupFolderNameList` |
| **設定読込** | `SetOptionFromSaveData()` |

**主要メソッド：**

```
// シーン遷移（非同期）
public async UniTask ChangeSceneAsync(int nextSceneIndex, bool useHeavyGCCollect = true)

// 保存された設定をランタイムオプションに読み込む
public void SetOptionFromSaveData()

// キャッシュされた楽曲メタデータをクリア
public void ClearSongsCache()
```

出典：

---

## データ永続化アーキテクチャ 

アプリケーションは3層の永続化アーキテクチャを使用しています：

```mermaid
flowchart TD

NO["NotesOption"]
DO["DisplayOption"]
VO["VolumeOption"]
JTO["JudgeTimeOption"]
GSP["GameSettingsPrefas Singleton<NotesOption> Singleton<DisplayOption> Singleton<VolumeOption> Singleton<JudgeTimeOption>"]
OSP["OnlineSettingsPrefas Singleton<OnlineSettings>"]
SDP["ScoreDataPrefas List<ScoreData>"]
JSON["JSONファイル (Application.persistentDataPath)"]
LITEDB["LiteDB (player.db)"]

NO -.->|"保存/読込"| GSP
DO -.->|"保存/読込"| GSP
VO -.->|"保存/読込"| GSP
JTO -.->|"保存/読込"| GSP
GSP -.->|"File<T>.Save()"| JSON
OSP -.->|"File<T>.Save()"| JSON
SDP -.->|"File<T>.Open()"| LITEDB
JSON -.->|"File<T>.Open()"| GSP
JSON -.-> OSP
LITEDB -.->|"書込"| SDP

subgraph ストレージ ["ストレージ"]
    JSON
    LITEDB
end

subgraph subGraph1 ["永続化レイヤー (Serializable シングルトン)"]
    GSP
    OSP
    SDP
end

subgraph subGraph0 ["ランタイム (GameManager)"]
    NO
    DO
    VO
    JTO
end
```

出典：

### オプションデータクラス 

以下のシリアライズ可能なクラスはプレイヤーの設定を保存します：

| クラス | 目的 | プロパティの例 |
| --- | --- | --- |
| `NotesOption` | ノート表示設定 | `NoteSpeed`, `NoteSize`, `NoteSkin`, `TouchSe` |
| `DisplayOption` | ビジュアル設定 | `TargetFrameRate`, `Dimmer`, `LaneCover`, `BeatBar` |
| `VolumeOption` | オーディオレベル | `MusicVolume`, `TouchSeVolume`, `SystemSeVolume` |
| `JudgeTimeOption` | タイミングウィンドウ | `JudgeWidth`, `LongEndRevision`, `FuzzyStartTime` |

**読込プロセス：**

`GameManager.Awake()`中に、オプションが永続ストレージから読み込まれます：

```c#
void Awake()
{
    base.Awake();
    SetOptionFromSaveData();  // GameSettingsPrefasから読込
    SetAudioDspBufferSize();
    Application.targetFrameRate = DisplayOption.TargetFrameRate;
    // ...
}
```


## シーン遷移メカニズム 

シーン遷移は`GameManager`によって管理され、同期モードと非同期モードの両方をサポートしています。

### 遷移フロー図 

```mermaid
sequenceDiagram
  participant p1 as ユーザーアクション
  participant p2 as Presenter
  participant p3 as GameManager
  participant p4 as SceneManager
  participant p5 as ガベージコレクター

  p1->>p2: ボタンクリック
  p2->>p3: ChangeSceneAsync(sceneIndex)
  p3->>p3: CurrentSceneIndex = sceneIndex
  p3->>p4: LoadSceneAsync(sceneIndex)
  p4-->>p3: シーン読込完了
  p3->>p4: Resources.UnloadUnusedAssets()
  p3->>p5: GC.Collect() x2
  p3-->>p2: シーン遷移完了
```

### シーンインデックス定数 

| シーン | インデックス | 典型的な遷移 |
| --- | --- | --- |
| Title | 0 | アプリ起動時、Select Musicから戻る |
| Select Music | 1 | Titleから（開始）、Resultから（リトライ） |
| Game | 2 | Select Musicから（プレイ） |
| Result | 3 | Gameから（楽曲完了） |

**シーン遷移メソッド：**

```c#
// リソースクリーンアップを伴う非同期遷移
public async UniTask ChangeSceneAsync(int nextSceneIndex, bool useHeavyGCCollect = true)
{    
    CurrentSceneIndex = nextSceneIndex;    
    await SceneManager.LoadSceneAsync(nextSceneIndex).ToUniTask();    
    await UniTask.SwitchToMainThread();    
    await Resources.UnloadUnusedAssets().ToUniTask();    
    GCCollect(useHeavyGCCollect);
}

// 同期遷移
public void ChangeScene(int nextSceneIndex, bool useHeavyGCCollect = true)
{    
    CurrentSceneIndex = nextSceneIndex;    
    SceneManager.LoadScene(nextSceneIndex);    
    Resources.UnloadUnusedAssets();    
    GCCollect(useHeavyGCCollect);
}
```

## ローカライゼーションシステム 

アプリケーションはUnityのLocalizationパッケージを使用し、型安全な文字列キーアクセスのための自動生成された定数クラスを利用します。

### ローカライゼーションアーキテクチャ 

```mermaid
flowchart TD

CSV["localize.csv (ソースデータ)"]
CONST["LocalizationConstant.cs (文字列キー定数)"]
TABLE_JA["DKLIKEStringTable_ja.asset"]
TABLE_EN["DKLIKEStringTable_en.asset"]
CODE["C#コード (UI、ダイアログなど)"]
LOC_MGR["LocalizeManager"]

CSV -.->|"自動生成"| CONST
CSV -.->|"インポート"| TABLE_JA
CSV -.->|"インポート"| TABLE_EN
CONST -.->|"使用"| CODE
CODE -.->|"クエリ"| LOC_MGR
```

**使用例：**

```c#
// マジック文字列の代わりに：
text.text = GetLocalizedString("CloseButton");
// 型安全な定数を使用：
text.text = GetLocalizedString(LocalizationConstant.CloseButton);
```

**利用可能なロケール：**

* 日本語 (ja): `DKLIKEStringTable_ja.asset`
* 英語 (en): `DKLIKEStringTable_en.asset`

---

## UIコンポーネント階層 

Titleシーンは、アプリケーション全体で使用される典型的なUI構成パターンを示しています：

```mermaid
flowchart TD

SCENE["Title.unity"]
LTS["TitleLifetimeScope"]
TSV["TitleSceneView"]
CANVAS["Canvas"]
TOP["TopPanel"]
MENU["MenuPanel"]
BG["BgImage"]
TITLE_IMG["TitleImage"]
TOUCH["TouchToStartImage"]
START["StartButton"]
MENU_BTN["MenuButton"]
CLOSE["CloseButton"]
MPC["MenuPanelController"]
NOTICE["NotificationButton"]
PACKAGE["PackageButton"]
SETTINGS["SettingButton"]
FAQ["FAQButton"]
LICENSE["LicenseButton"]
BACKUP["BackupButton"]

SCENE -.-> LTS
LTS -.-> TSV
TSV -.-> CANVAS
CANVAS -.-> TOP
CANVAS -.-> MENU
TOP -.-> BG
TOP -.-> TITLE_IMG
TOP -.-> TOUCH
TOP -.-> START
TOP -.-> MENU_BTN
MENU -.-> CLOSE
MENU -.-> MPC
MPC -.-> NOTICE
MPC -.-> PACKAGE
MPC -.-> SETTINGS
MPC -.-> FAQ
MPC -.-> LICENSE
MPC -.-> BACKUP
```

出典：

 

 

---

## 定数と設定 

アプリケーション全体の定数は専用のクラスで定義されています：

### 定数クラス構造 

| 定数クラス | 目的 | 定数の例 |
| --- | --- | --- |
| `Constant` | ゲームロジック定数 | `Note.BEAT_DISTANCE`, `JudgeTime.PERFECT_TIME`, `RankBoundary.S_PLUS` |
| `LocalizationConstant` | 文字列テーブルキー | `CloseButton`, `GameSettingButton`, `PlayButton` |
| `ResourcesName` | リソースパス | プレハブパス、アドレサブルキー |

**定数の例：**

```c#
// Constant.cs
public static class Note
{    
    public const float BEAT_DISTANCE = 1.5f;
    public const float LANE_DISTANCE = 0.5f;
    public const int LANE_COUNT = 7;
}

public static class JudgeTime
{    
    public const float PERFECT_TIME = 0.030f;    
    public const float BRILLIANT_TIME = 0.050f;    
    public const float GREAT_TIME = 0.100f;
}

public static class App
{    
    public static readonly string VERSION_NAME = "#95";    
    public static readonly string HOMEPAGE_URL = "/";
}
```

## まとめ 

DKLIKEアプリケーションのアーキテクチャは、以下の主要な原則に従っています：

1. **シーンベースの構成**：明確な責務を持つ4つのメインシーン（Title、SelectMusic、Game、Result）
2. **MVPパターン**：各シーンが関心事の分離のために`LifetimeScope → Presenter → View`構造を使用
3. **シングルトン状態**：`GameManager`がシーン間で永続化し、グローバルオプションと状態を管理
4. **型安全な永続化**：自動保存/読込のための`Serializable<T>`パターンを持つシリアライズ可能なオプションクラス
5. **ローカライゼーション**：型安全性のための自動生成された定数クラスを持つUnity Localizationパッケージ
6. **リソース管理**：ガベージコレクションを伴うシーン遷移中の適切なクリーンアップ

このアーキテクチャは、シーン間の明確な境界、UI管理のための一貫したパターン、およびユーザー設定とゲームデータの信頼性の高い永続化を提供します。

### On this page

* [アプリケーション構造](#2-)
* [目的と範囲](#2--1)
* [シーン構造](#2--2)
* [シーンフロー図](#2--3)
* [MVPパターン構造](#2-mvp)
* [MVPクラス階層](#2-mvp-1)
* [Presenterライフサイクル](#2-presenter)
* [グローバルステート管理](#2--4)
* [GameManagerシングルトン](#2-gamemanager)
* [GameManagerの主要な責務](#2-gamemanager-1)
* [データ永続化アーキテクチャ](#2--5)
* [オプションデータクラス](#2--6)
* [シーン遷移メカニズム](#2--7)
* [遷移フロー図](#2--8)
* [シーンインデックス定数](#2--9)
* [ローカライゼーションシステム](#2--10)
* [ローカライゼーションアーキテクチャ](#2--11)
* [UIコンポーネント階層](#2-ui)
* [定数と設定](#2--12)
* [定数クラス構造](#2--13)
* [まとめ](#2--14)

